<?php
/*
Plugin Name: Post Archive Index by Web321
Description: A plugin to create an index/archive of all posts.
plugin URI: https://web321.co/plugins/
Donate: https://www.paypal.com/paypalme/web321co/10/
Version: 1.0
Author: dewolfe001
Author URI: https://web321.co/
License: GPLv2 or later
*/

defined('ABSPATH') || exit;

define('WAI_BLOCK_NAME', plugin_basename(__FILE__));
define('WAI_BLOCK_DIR', plugin_dir_path(__FILE__));
define('WAI_BLOCK_URL', plugin_dir_url(__FILE__));
define('WAI_BLOCK_VERSION', '1.0');

add_filter('plugin_row_meta', 'wai_block_row_meta', 10, 2);

// Donation link
function wai_block_row_meta($links, $file) {
    if (WAI_BLOCK_NAME == $file) {
        $row_meta = [
          'donate' => '<a href="' . esc_url('https://www.paypal.com/paypalme/web321co/10/') . '" target="_blank" aria-label="' . esc_attr__('Donate', 'wai-block') . '" >' . esc_html__('Donate', 'wai-block') . '</a>'
        ];
        return array_merge($links, $row_meta);
    }
    return (array) $links;
}

// Enqueue scripts for block and editor
function wai_enqueue_custom_scripts() {
    wp_enqueue_script(
        'wai-block-js',
        WAI_BLOCK_URL . 'assets/block.js',
        array('wp-blocks', 'wp-element', 'wp-editor', 'wp-components', 'wp-data'),
        filemtime(WAI_BLOCK_DIR . 'assets/block.js')
    );
    wp_enqueue_script(
        'wai-ajax-js',
        WAI_BLOCK_URL . 'assets/ajax.js',
        array('jquery'),
        filemtime(WAI_BLOCK_DIR . 'assets/ajax.js')
    );

    wp_localize_script('wai-ajax-js', 'wai_ajax_object', array(
        'ajax_url' => admin_url('admin-ajax.php')
    ));
}

add_action('enqueue_block_editor_assets', 'wai_enqueue_custom_scripts');
add_action('wp_enqueue_scripts', 'wai_enqueue_custom_scripts');

// Handle AJAX request to get current page categories and tags
function wai_get_current_page_terms() {
    $post_id = intval($_POST['post_id']);
    if ($post_id) {
        $categories = wp_get_post_categories($post_id, array('fields' => 'names'));
        $tags = wp_get_post_tags($post_id, array('fields' => 'names'));
        wp_send_json_success(array('categories' => $categories, 'tags' => $tags));
    } else {
        wp_send_json_error();
    }
}
add_action('wp_ajax_wai_get_current_page_terms', 'wai_get_current_page_terms');

// Handle AJAX search request
function wai_search_posts() {
    $search_query = sanitize_text_field($_POST['s']);

    // WP Query to fetch posts based on search query
    $args = array(
        'post_type' => 'any',
        'posts_per_page' => 10,
        's' => $search_query,
    );

    $query = new WP_Query($args);

    if ($query->have_posts()) {
        ob_start();
        echo '<ul>';
        while ($query->have_posts()) {
            $query->the_post();
            echo '<li><a href="' . get_permalink() . '">' . get_the_title() . '</a></li>';
        }
        echo '</ul>';
        wp_reset_postdata();
        $content = ob_get_clean();
        wp_send_json_success($content);
    } else {
        wp_send_json_error();
    }
}
add_action('wp_ajax_wai_search_posts', 'wai_search_posts');
add_action('wp_ajax_nopriv_wai_search_posts', 'wai_search_posts');

// Render the block/widget with categories and tags
function wai_render_custom_block($attributes) {
    if ((isset($attributes['useCurrentPageTerms'])) && ($attributes['useCurrentPageTerms'])) {
        global $post;
        $categories = wp_get_post_categories($post->ID, array('fields' => 'names'));
        $tags = wp_get_post_tags($post->ID, array('fields' => 'names'));
    } else {
        $categories = $attributes['categories'];
        $tags = $attributes['tags'];
    }

    // WP Query to fetch posts based on categories and tags
    $args = array(
        'post_type' => 'post',
        'posts_per_page' => -1
    );

    if (is_array($categories)) {
        $categories = array_filter($categories);
    }

    if (is_array($tags)) {
        $tags = array_filter($tags);
    }
    
    // Construct tax_query only if we have valid terms
    $tax_query = []; // array('relation' => 'AND');
    if (!empty($categories)) {
        foreach ($categories as $tag) {
            $term_exists = term_exists($tag, 'post_tag');
            if (!$term_exists) {
                unset($categories[$tag]);                           
            }
        }

        $tax_query[] = array(
            'taxonomy' => 'category',
            'field'    => 'name',
            'terms'    => $categories,
        );
    }
    if (!empty($tags)) {
        $post_tags = $tags;
        foreach ($post_tags as $tag) {
            $term_exists = term_exists($tag, 'post_tag');
            if (!$term_exists) {
                unset($post_tags[$tag]);                           
            }
        }
        $tax_query[] = array(
            'taxonomy' => 'post_tag',
            'field'    => 'name',
            'terms'    => $post_tags,
        ); 

        foreach ($tags as $tag) {
            $term_exists = term_exists($tag, 'tags');
            if (!$term_exists) {
                unset($tags[$tag]);                           
            }
        }        
        $tax_query[] = array(
            'taxonomy' => 'tags',
            'field'    => 'name',
            'terms'    => $tags,
        );         
    }

    // WP Query to fetch posts based on categories and tags
    $args = array(
        'post_type' => 'any', // Query all post types
        'posts_per_page' => 10,
    );

    if (count($tax_query) > 0) {
        $args['tax_query'] = $tax_query;
    }

    add_filter('posts_where', 'wai_remove_zero_equals_one', 10, 2);
    $query = new WP_Query($args);
    
    ob_start();
    ?>
    <div class="custom-block">
        <h4>Search Posts</h4>
        <form role="search" method="get" class="search-form" action="<?php echo home_url('/'); ?>">
            <label>
                <span class="screen-reader-text"><?php echo _x('Search for:', 'label'); ?></span>
                <input type="search" class="search-field" placeholder="<?php echo esc_attr_x('Search …', 'placeholder'); ?>" value="<?php echo get_search_query(); ?>" name="s" />
            </label>
            <button type="submit" class="search-submit"><?php echo esc_html_x('Search', 'submit button'); ?></button>
        </form>
        <div class="search-results">
            <?php
            if ($query->have_posts()) {
                echo '<ul>';
                while ($query->have_posts()) {
                    $query->the_post();
                    echo '<li><a href="' . get_permalink() . '">' . get_the_title() . '</a></li>';
                }
                echo '</ul>';
            } else {
                echo '<p>No posts found.</p>';
            }
            ?>
        </div>
    </div>
    <?php
    wp_reset_postdata();
    return ob_get_clean();
}

function wai_remove_zero_equals_one($where, $wp_query) {
    $where = str_replace("0 = 1 ", "1 = 1", $where);
    return $where;
}

// Register the block type with the server-side rendering callback
function wai_register_blocks() {
    register_block_type('wai/block', array(
        'render_callback' => 'wai_render_custom_block',
    ));
}
add_action('init', 'wai_register_blocks');

// Register the widget
class WAI_Category_Tag_Widget extends WP_Widget {
    function __construct() {
        parent::__construct(
            'wai_category_tag_widget',
            __('WAI Category Tag Widget', 'text_domain'),
            array('description' => __('A widget to display categories and tags.', 'text_domain'))
        );
    }

    public function widget($args, $instance) {
        global $post;
        $categories = !empty($instance['use_current_page_terms']) ? wp_get_post_categories($post->ID, array('fields' => 'names')) : explode(',', $instance['categories']);
        $tags = !empty($instance['use_current_page_terms']) ? wp_get_post_tags($post->ID, array('fields' => 'names')) : explode(',', $instance['tags']);

        // WP Query to fetch posts based on categories and tags
        $query_args = array(
            'post_type' => 'any',
            'posts_per_page' => -1
        );
        
        // Filter out empty values
        $categories = array_filter($categories);
        $tags = array_filter($tags);
    
        // Construct tax_query only if we have valid terms
        $tax_query = array('relation' => 'AND');
        if (!empty($categories)) {
            $tax_query[] = array(
                'taxonomy' => 'category',
                'field'    => 'slug',
                'terms'    => $categories,
            );
        }
        if (!empty($tags)) {
            $tax_query[] = array(
                'taxonomy' => 'post_tag',
                'field'    => 'slug',
                'terms'    => $tags,
            );
        }
    
        // WP Query to fetch posts based on categories and tags
        $query_args = array(
            'post_type' => 'any', // Query all post types
            'posts_per_page' => 10,
            'tax_query' => count($tax_query) > 1 ? $tax_query : array(), // Only add if we have valid queries
        );
    
        $query = new WP_Query($query_args);

        echo $args['before_widget'];
        if (!empty($instance['title'])) {
            echo $args['before_title'] . apply_filters('widget_title', $instance['title']) . $args['after_title'];
        }
        ?>
        <div class="custom-widget">
            <h4>Search Posts</h4>
            <form role="search" method="get" class="search-form" action="<?php echo home_url('/'); ?>">
                <label>
                    <span class="screen-reader-text"><?php echo _x('Search for:', 'label'); ?></span>
                    <input type="search" class="search-field" placeholder="<?php echo esc_attr_x('Search …', 'placeholder'); ?>" value="<?php echo get_search_query(); ?>" name="s" />
                </label>
                <button type="submit" class="search-submit"><?php echo esc_html_x('Search', 'submit button'); ?></button>
            </form>
            <div class="search-results">
                <?php
                if ($query->have_posts()) {
                    echo '<ul>';
                    while ($query->have_posts()) {
                        $query->the_post();
                        echo '<li><a href="' . get_permalink() . '">' . get_the_title() . '</a></li>';
                    }
                    echo '</ul>';
                    wp_reset_postdata();
                } else {
                    echo '<p>No posts found.</p>';
                }
                ?>
            </div>
        </div>
        <?php
        echo $args['after_widget'];
    }

    public function form($instance) {
        $title = !empty($instance['title']) ? $instance['title'] : '';
        $categories = !empty($instance['categories']) ? $instance['categories'] : '';
        $tags = !empty($instance['tags']) ? $instance['tags'] : '';
        $use_current_page_terms = !empty($instance['use_current_page_terms']) ? $instance['use_current_page_terms'] : false;
        ?>
        <p>
            <label for="<?php echo $this->get_field_id('title'); ?>"><?php _e('Title:'); ?></label>
            <input class="widefat" id="<?php echo $this->get_field_id('title'); ?>" name="<?php echo $this->get_field_name('title'); ?>" type="text" value="<?php echo esc_attr($title); ?>">
        </p>
        <p>
            <input class="checkbox" type="checkbox" <?php checked($use_current_page_terms); ?> id="<?php echo $this->get_field_id('use_current_page_terms'); ?>" name="<?php echo $this->get_field_name('use_current_page_terms'); ?>" />
            <label for="<?php echo $this->get_field_id('use_current_page_terms'); ?>"><?php _e('Use current page categories and tags'); ?></label>
        </p>
        <p>
            <label for="<?php echo $this->get_field_id('categories'); ?>"><?php _e('Categories (comma separated):'); ?></label>
            <input class="widefat" id="<?php echo $this->get_field_id('categories'); ?>" name="<?php echo $this->get_field_name('categories'); ?>" type="text" value="<?php echo esc_attr($categories); ?>" <?php disabled($use_current_page_terms); ?>>
        </p>
        <p>
            <label for="<?php echo $this->get_field_id('tags'); ?>"><?php _e('Tags (comma separated):'); ?></label>
            <input class="widefat" id="<?php echo $this->get_field_id('tags'); ?>" name="<?php echo $this->get_field_name('tags'); ?>" type="text" value="<?php echo esc_attr($tags); ?>" <?php disabled($use_current_page_terms); ?>>
        </p>
        <?php
    }

    public function update($new_instance, $old_instance) {
        $instance = array();
        $instance['title'] = (!empty($new_instance['title'])) ? strip_tags($new_instance['title']) : '';
        $instance['categories'] = (!empty($new_instance['categories'])) ? strip_tags($new_instance['categories']) : '';
        $instance['tags'] = (!empty($new_instance['tags'])) ? strip_tags($new_instance['tags']) : '';
        $instance['use_current_page_terms'] = !empty($new_instance['use_current_page_terms']) ? (bool)$new_instance['use_current_page_terms'] : false;

        return $instance;
    }
}

function wai_register_category_tag_widget() {
    register_widget('WAI_Category_Tag_Widget');
}
add_action('widgets_init', 'wai_register_category_tag_widget');